//
//  ColorWheel.swift
//  Do It
//
//  Created by Jim Dovey on 2/4/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

fileprivate struct LoupeLocationPreferenceKey: PreferenceKey {
    typealias Value = Anchor<CGPoint>?
    static var defaultValue: Anchor<CGPoint>? = nil
    static func reduce(
        value: inout Anchor<CGPoint>?,
        nextValue: () -> Anchor<CGPoint>?
    ) {
        if value == nil {
            value = nextValue()
        }
    }
}

fileprivate struct BrightnessBar<Value: ColorInfo>: View {
    @Binding var color: Value

    var gradient: Gradient {
        let (h, s, _) = color.hsb
        return Gradient(colors: [
            Color(hue: h, saturation: s, brightness: 1),
            Color(hue: h, saturation: s, brightness: 0)
        ])
    }

    func selectionOffset(_ proxy: GeometryProxy) -> CGSize {
        CGSize(width: 0,
               height: CGFloat(1.0-color.brightness) * proxy.size.height - 5)
    }

    var body: some View {
        GeometryReader { proxy in
            ZStack(alignment: .top) {
                LinearGradient(gradient: self.gradient,
                               startPoint: .top,
                               endPoint: .bottom)
                    .border(Color.white)

                self.color.uiColor
                    .border(Color.white)
                    .frame(height: 10)
                    .offset(self.selectionOffset(proxy))
            }
            .gesture(DragGesture(minimumDistance: 0).onChanged {
                let value = 1.0 - Double($0.location.y / proxy.size.height)
                self.color.brightness = min(max(0.0, value), 1.0)
            })
            .accessibility(label: Text("Brightness"))
            .accessibility(value: Text("\(self.color.brightness as NSNumber, formatter: Formatters.percentage)"))
            .accessibilityAdjustableAction { direction in
                switch direction {
                case .increment:
                    let current = self.color.brightness
                    self.color.brightness = min(current + 0.1, 1.0)
                case .decrement:
                    let current = self.color.brightness
                    self.color.brightness = max(current - 0.1, 0.0)
                @unknown default:
                    break
                }
            }
        }
    }
}

struct ColorWheel<Value: ColorInfo>: View {
    @Binding var color: Value
    
    @State private var dragging = false
    @State private var loupeLocation: CGPoint = .zero

    private var wheelGradient: AngularGradient {
        let (_, _, b) = color.hsb
        let stops: [Gradient.Stop] = stride(from: 0.0, through: 1.0, by: 0.01).map {
            Gradient.Stop(color: Color(hue: $0, saturation: 1, brightness: b),
                          location: CGFloat($0))
        }
        let gradient = Gradient(stops: stops)
        return AngularGradient(gradient: gradient, center: .center,
                               angle: .degrees(360))
    }

    private func fadeGradient(radius: CGFloat) -> RadialGradient {
        let (_, _, b) = color.hsb
        let fadeColor = Color(hue: 0, saturation: 0, brightness: b)
        let gradient = Gradient(colors: [fadeColor, fadeColor.opacity(0)])
        return RadialGradient(gradient: gradient, center: .center,
                              startRadius: 0, endRadius: radius)
    }

    var body: some View {
        HStack(spacing: 16) {
            GeometryReader { proxy in
                ZStack {
                    self.wheelGradient
                        .overlay(self.fadeGradient(radius: proxy.size.width/2))
                        .clipShape(Circle())
                        .overlay(Circle().stroke(Color.white))

                    if !self.dragging {
                        Rectangle()
                            .fill(self.color.uiColor)
                            .overlay(Rectangle().stroke(Color.white, lineWidth: 1))
                            .frame(width: 16, height: 16)
                            .offset(HSB.unitOffset(for: self.color, within: proxy.size))
                    }
                }
                .gesture(
                    DragGesture(minimumDistance: 0).onChanged {
                        self.dragging = true
                        self.loupeLocation = $0.location
                            .boundedInCircle(radius: proxy.size.width/2)
                        self.assignColor(at: self.loupeLocation, in: proxy)
                    }
                    .onEnded { _ in
                        self.dragging = false
                        self.assignColor(at: self.loupeLocation, in: proxy)
                    }
                )
                .anchorPreference(key: LoupeLocationPreferenceKey.self,
                                  value: .point(self.loupeLocation),
                                  transform: { $0 })
            }
            .aspectRatio(contentMode: .fit)
            .modifier(DoubleShadow())
            .overlayPreferenceValue(LoupeLocationPreferenceKey.self) { anchor in
                GeometryReader { geometry in
                    self.buildLoupe(geometry, anchor)
                        .opacity(self.dragging ? 1 : 0)
                }
            }
            .accessibility(hidden: true)

            BrightnessBar(color: self.$color)
                .padding(.vertical, 30)
                .frame(maxWidth: 30)
                .modifier(DoubleShadow())
                .zIndex(-1)
        }
        .aspectRatio(1.125, contentMode: .fit)
    }

    private func buildLoupe(
        _ geometry: GeometryProxy,
        _ anchor: Anchor<CGPoint>?
    ) -> some View {
        let location = anchor != nil ? geometry[anchor!] : .zero
        let unitLocation = location.centeredUnit(within: geometry.size)

        return Circle()
            .fill(HSB.uiColor(at: unitLocation, basedOn: color))
            .overlay(Circle().stroke(Color.white, lineWidth: 1))
            .frame(width: 70, height: 70)
            .modifier(DoubleShadow())
            .offset(x: location.x - 35, y: location.y - 35)
    }

    private func assignColor(at location: CGPoint, in geometry: GeometryProxy) {
        let unitLocation = location.centeredUnit(within: geometry.size)
        HSB.updateColor(&color, at: unitLocation)
    }
}

struct ColorWheel_Previews: PreviewProvider {
    static var previews: some View {
        StatefulPreviewWrapper(TodoItemList.Color.purple) { binding in
            VStack {
                ColorWheel(color: binding)

                RoundedRectangle(cornerRadius: 12)
                    .fill(binding.wrappedValue.uiColor)
                    .overlay(RoundedRectangle(cornerRadius: 12)
                        .stroke(Color.white))
                    .frame(width: 300, height: 60)
                    .modifier(DoubleShadow())
                    .padding(.top)
                    .zIndex(-1)
            }
        }
    }
}
